<?php
defined('_JEXEC') or die;

$props = $props ?? [];
$attrs = $attrs ?? [];

// Internal unique token for this element instance (CSS + gradient IDs)
$uid = 'sc-' . substr(md5(json_encode($props) . uniqid('', true)), 0, 12);
$uid = preg_replace('~[^a-zA-Z0-9\-_]~', '', $uid);

$sanitizeColor = function ($value, string $fallback = '#ffffff'): string {
    $v = trim((string) $value);
    if ($v === '') {
        return $fallback;
    }
    if (preg_match('/^#[0-9a-fA-F]{3}([0-9a-fA-F]{1})?([0-9a-fA-F]{2})?([0-9a-fA-F]{2})?$/', $v)) {
        return $v;
    }
    $lc = strtolower($v);
    if (in_array($lc, ['transparent', 'currentcolor'], true)) {
        return $lc === 'currentcolor' ? 'currentColor' : 'transparent';
    }
    if (preg_match('/^var\(--[a-zA-Z0-9_-]+\)$/', $v)) {
        return $v;
    }
    return $fallback;
};

$sanitizeHex = function ($value, string $fallback = '#000000'): string {
    $v = trim((string) $value);
    if ($v === '') {
        return $fallback;
    }
    if (preg_match('/^#[0-9a-fA-F]{3}$/', $v) || preg_match('/^#[0-9a-fA-F]{6}$/', $v)) {
        return strtolower($v);
    }
    return $fallback;
};

$hexToRgba = function (string $hex, float $opacity): string {
    $hex = ltrim($hex, '#');
    if (strlen($hex) === 3) {
        $r = hexdec(str_repeat($hex[0], 2));
        $g = hexdec(str_repeat($hex[1], 2));
        $b = hexdec(str_repeat($hex[2], 2));
    } else {
        $r = hexdec(substr($hex, 0, 2));
        $g = hexdec(substr($hex, 2, 2));
        $b = hexdec(substr($hex, 4, 2));
    }
    $opacity = max(0.0, min(1.0, $opacity));
    // Use dot decimal and avoid locale issues
    $a = rtrim(rtrim(number_format($opacity, 3, '.', ''), '0'), '.');
    if ($a === '') { $a = '0'; }
    return "rgba($r,$g,$b,$a)";
};


$top    = !empty($props['top_enable']);
$bottom = !empty($props['bottom_enable']);

$topColor    = $sanitizeColor($props['top_color'] ?? '', '#ffffff');
$bottomColor = $sanitizeColor($props['bottom_color'] ?? '', '#ffffff');

$topGradientCode    = trim((string) ($props['top_gradient_code'] ?? ''));
$bottomGradientCode = trim((string) ($props['bottom_gradient_code'] ?? ''));
$topHasGradient     = $topGradientCode !== '';
$bottomHasGradient  = $bottomGradientCode !== '';

$topCustomSvg    = trim((string) ($props['top_custom_svg'] ?? ''));
$bottomCustomSvg = trim((string) ($props['bottom_custom_svg'] ?? ''));
$topHasCustom    = $topCustomSvg !== '';
$bottomHasCustom = $bottomCustomSvg !== '';

// Shadow (drop-shadow on SVG)
$topShadowOn = !empty($props['top_shadow_enable']);
$bottomShadowOn = !empty($props['bottom_shadow_enable']);

$topShadowX = (int) ($props['top_shadow_x'] ?? 0);
$topShadowY = (int) ($props['top_shadow_y'] ?? 10);
$topShadowBlur = max(0, (int) ($props['top_shadow_blur'] ?? 20));
$topShadowOpacity = (float) ($props['top_shadow_opacity'] ?? 0.35);

$bottomShadowX = (int) ($props['bottom_shadow_x'] ?? 0);
$bottomShadowY = (int) ($props['bottom_shadow_y'] ?? -10);
$bottomShadowBlur = max(0, (int) ($props['bottom_shadow_blur'] ?? 20));
$bottomShadowOpacity = (float) ($props['bottom_shadow_opacity'] ?? 0.35);

$topShadowColorHex = $sanitizeHex($props['top_shadow_color'] ?? '', '#000000');
$bottomShadowColorHex = $sanitizeHex($props['bottom_shadow_color'] ?? '', '#000000');


$topShadowCss = 'none';
if ($topShadowOn) {
    $topShadowCss = 'drop-shadow(' . $topShadowX . 'px ' . $topShadowY . 'px ' . $topShadowBlur . 'px ' . $hexToRgba($topShadowColorHex, $topShadowOpacity) . ')';
}
$bottomShadowCss = 'none';
if ($bottomShadowOn) {
    $bottomShadowCss = 'drop-shadow(' . $bottomShadowX . 'px ' . $bottomShadowY . 'px ' . $bottomShadowBlur . 'px ' . $hexToRgba($bottomShadowColorHex, $bottomShadowOpacity) . ')';
}

$topShadowRgba = $hexToRgba($topShadowColorHex, $topShadowOpacity);
$bottomShadowRgba = $hexToRgba($bottomShadowColorHex, $bottomShadowOpacity);

$topShadowFilter = $topShadowOn ? ("drop-shadow({$topShadowX}px {$topShadowY}px {$topShadowBlur}px {$topShadowRgba})") : 'none';
$bottomShadowFilter = $bottomShadowOn ? ("drop-shadow({$bottomShadowX}px {$bottomShadowY}px {$bottomShadowBlur}px {$bottomShadowRgba})") : 'none';


$heightVal  = (float) ($props['height_value'] ?? 0);
$heightVal  = max(0, $heightVal);
$heightUnit = (string) ($props['height_unit'] ?? 'px');
if (!in_array($heightUnit, ['px', 'vh', 'rem'], true)) {
    $heightUnit = 'px';
}

$bg = trim((string) ($props['background'] ?? ''));

$overflowHidden = !empty($props['overflow_hidden']);
$zIndex         = (int) ($props['z_index'] ?? 0);

$wrap = $this->el('div', [
    'class' => [
        'hpb-shapes-curves',
        $uid,
        $props['class'] ?? null,
    ],
    // Let the user set the ID via Advanced. If empty, no id attribute is printed.
    'id' => trim((string) ($props['id'] ?? '')) ?: null,
]);

$shapes = [
  'wave1' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V46.29c47.79,22.2,103.59,29,158,17.39C230.36,48.27,284,6.35,339.11,0c54.14-6.25,104.73,24.11,158.72,44.79C585.09,74.49,644,92.7,705.67,92.65c61.33-.05,120.09-18.81,180.59-38.06C990.62,29.94,1108.77,0,1200,16.24V0Z'],
  'wave2' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V27.35c74.19,40.68,156.23,48.63,242,29.08C350.73,33.89,434.78-6.77,525.28,1.85c63.78,6.1,117.43,38.16,180.3,59.9C794,92.72,865.43,102.72,936,94.87c93.11-10.36,178.31-50.74,264-77.51V0Z'],
  // cleaner, more symmetric waves
  'wave3' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V60C150,0,350,120,600,60C850,0,1050,120,1200,60V0Z'],
  'wave4' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V80C200,140,400,20,600,80C800,140,1000,20,1200,80V0Z'],
  'curve1' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V120H1200V0C960,80,720,120,480,120S240,80,0,0Z'],
  'curve2' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V120H1200V0C1030,65,840,100,600,100S170,65,0,0Z'],
  // arcs / bowls
  'arc1'  => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V100Q600,0,1200,100V0Z'],
  'arc2'  => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V120Q600,30,1200,120V0Z'],
  'tilt'  => ['viewBox' => '0 0 1200 120', 'path' => 'M1200,0V120H0V60Z'],
  'diagonal' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V120H1200V0H0L0,60Z'],
  'triangle' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V120H1200V0L600,120Z'],
  'split' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V90L600,120L1200,90V0Z'],
  'zigzag' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V120H1200V0L1050,120L900,0L750,120L600,0L450,120L300,0L150,120Z'],
  'peaks' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V90L150,120L300,90L450,120L600,90L750,120L900,90L1050,120L1200,90V0Z'],
  'spikes' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V80L100,120L200,80L300,120L400,80L500,120L600,80L700,120L800,80L900,120L1000,80L1100,120L1200,80V0Z'],
  'steps' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V90H150V110H300V90H450V110H600V90H750V110H900V90H1050V110H1200V0Z'],
  'drip'  => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V60C100,100,200,20,300,60C400,100,500,20,600,60C700,100,800,20,900,60C1000,100,1100,20,1200,60V0Z'],

  // --- Extra shapes (pack 50) ---
  // Waves 5-20
  'wave5' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V70C100,70,100,98,200,98C300,98,300,70,400,70C500,70,500,42,600,42C700,42,700,70,800,70C900,70,900,98,1000,98C1100,98,1100,70,1200,70V0Z'],
  'wave6' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V65C75,65,75,90,150,90C225,90,225,65,300,65C375,65,375,40,450,40C525,40,525,65,600,65C675,65,675,90,750,90C825,90,825,65,900,65C975,65,975,40,1050,40C1125,40,1125,65,1200,65V0Z'],
  'wave7' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V62C60,62,60,84,120,84C180,84,180,62,240,62C300,62,300,40,360,40C420,40,420,62,480,62C540,62,540,84,600,84C660,84,660,62,720,62C780,62,780,40,840,40C900,40,900,62,960,62C1020,62,1020,84,1080,84C1140,84,1140,62,1200,62V0Z'],
  'wave8' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V58C50,58,50,76,100,76C150,76,150,58,200,58C250,58,250,40,300,40C350,40,350,58,400,58C450,58,450,76,500,76C550,76,550,58,600,58C650,58,650,40,700,40C750,40,750,58,800,58C850,58,850,76,900,76C950,76,950,58,1000,58C1050,58,1050,40,1100,40C1150,40,1150,58,1200,58V0Z'],
  'wave9' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V55C42.86,55,42.86,71,85.71,71C128.57,71,128.57,55,171.43,55C214.29,55,214.29,39,257.14,39C300,39,300,55,342.86,55C385.71,55,385.71,71,428.57,71C471.43,71,471.43,55,514.29,55C557.14,55,557.14,39,600,39C642.86,39,642.86,55,685.71,55C728.57,55,728.57,71,771.43,71C814.29,71,814.29,55,857.14,55C900,55,900,39,942.86,39C985.71,39,985.71,55,1028.57,55C1071.43,55,1071.43,71,1114.29,71C1157.14,71,1157.14,55,1200,55V0Z'],
  'wave10' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V52C37.50,52,37.50,66,75,66C112.50,66,112.50,52,150,52C187.50,52,187.50,38,225,38C262.50,38,262.50,52,300,52C337.50,52,337.50,66,375,66C412.50,66,412.50,52,450,52C487.50,52,487.50,38,525,38C562.50,38,562.50,52,600,52C637.50,52,637.50,66,675,66C712.50,66,712.50,52,750,52C787.50,52,787.50,38,825,38C862.50,38,862.50,52,900,52C937.50,52,937.50,66,975,66C1012.50,66,1012.50,52,1050,52C1087.50,52,1087.50,38,1125,38C1162.50,38,1162.50,52,1200,52V0Z'],
  'wave11' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V78C100,78,100,114,200,114C300,114,300,78,400,78C500,78,500,42,600,42C700,42,700,78,800,78C900,78,900,114,1000,114C1100,114,1100,78,1200,78V0Z'],
  'wave12' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V74C75,74,75,106,150,106C225,106,225,74,300,74C375,74,375,42,450,42C525,42,525,74,600,74C675,74,675,106,750,106C825,106,825,74,900,74C975,74,975,42,1050,42C1125,42,1125,74,1200,74V0Z'],
  'wave13' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V70C60,70,60,100,120,100C180,100,180,70,240,70C300,70,300,40,360,40C420,40,420,70,480,70C540,70,540,100,600,100C660,100,660,70,720,70C780,70,780,40,840,40C900,40,900,70,960,70C1020,70,1020,100,1080,100C1140,100,1140,70,1200,70V0Z'],
  'wave14' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V66C50,66,50,92,100,92C150,92,150,66,200,66C250,66,250,40,300,40C350,40,350,66,400,66C450,66,450,92,500,92C550,92,550,66,600,66C650,66,650,40,700,40C750,40,750,66,800,66C850,66,850,92,900,92C950,92,950,66,1000,66C1050,66,1050,40,1100,40C1150,40,1150,66,1200,66V0Z'],
  'wave15' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V64C42.86,64,42.86,88,85.71,88C128.57,88,128.57,64,171.43,64C214.29,64,214.29,40,257.14,40C300,40,300,64,342.86,64C385.71,64,385.71,88,428.57,88C471.43,88,471.43,64,514.29,64C557.14,64,557.14,40,600,40C642.86,40,642.86,64,685.71,64C728.57,64,728.57,88,771.43,88C814.29,88,814.29,64,857.14,64C900,64,900,40,942.86,40C985.71,40,985.71,64,1028.57,64C1071.43,64,1071.43,88,1114.29,88C1157.14,88,1157.14,64,1200,64V0Z'],
  'wave16' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V60C33.33,60,33.33,80,66.67,80C100,80,100,60,133.33,60C166.67,60,166.67,40,200,40C233.33,40,233.33,60,266.67,60C300,60,300,80,333.33,80C366.67,80,366.67,60,400,60C433.33,60,433.33,40,466.67,40C500,40,500,60,533.33,60C566.67,60,566.67,80,600,80C633.33,80,633.33,60,666.67,60C700,60,700,40,733.33,40C766.67,40,766.67,60,800,60C833.33,60,833.33,80,866.67,80C900,80,900,60,933.33,60C966.67,60,966.67,40,1000,40C1033.33,40,1033.33,60,1066.67,60C1100,60,1100,80,1133.33,80C1166.67,80,1166.67,60,1200,60V0Z'],
  'wave17' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V58C30,58,30,76,60,76C90,76,90,58,120,58C150,58,150,40,180,40C210,40,210,58,240,58C270,58,270,76,300,76C330,76,330,58,360,58C390,58,390,40,420,40C450,40,450,58,480,58C510,58,510,76,540,76C570,76,570,58,600,58C630,58,630,40,660,40C690,40,690,58,720,58C750,58,750,76,780,76C810,76,810,58,840,58C870,58,870,40,900,40C930,40,930,58,960,58C990,58,990,76,1020,76C1050,76,1050,58,1080,58C1110,58,1110,40,1140,40C1170,40,1170,58,1200,58V0Z'],
  'wave18' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V80C75,80,75,120,150,120C225,120,225,80,300,80C375,80,375,40,450,40C525,40,525,80,600,80C675,80,675,120,750,120C825,120,825,80,900,80C975,80,975,40,1050,40C1125,40,1125,80,1200,80V0Z'],
  'wave19' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V78C60,78,60,116,120,116C180,116,180,78,240,78C300,78,300,40,360,40C420,40,420,78,480,78C540,78,540,116,600,116C660,116,660,78,720,78C780,78,780,40,840,40C900,40,900,78,960,78C1020,78,1020,116,1080,116C1140,116,1140,78,1200,78V0Z'],
  'wave20' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V76C50,76,50,110,100,110C150,110,150,76,200,76C250,76,250,42,300,42C350,42,350,76,400,76C450,76,450,110,500,110C550,110,550,76,600,76C650,76,650,42,700,42C750,42,750,76,800,76C850,76,850,110,900,110C950,110,950,76,1000,76C1050,76,1050,42,1100,42C1150,42,1150,76,1200,76V0Z'],
  // Saw / Teeth 1-12
  'saw1' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V80L200,120L400,80L600,120L800,80L1000,120L1200,80V0Z'],
  'saw2' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V78L150,120L300,78L450,120L600,78L750,120L900,78L1050,120L1200,78V0Z'],
  'saw3' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V75L120,120L240,75L360,120L480,75L600,120L720,75L840,120L960,75L1080,120L1200,75V0Z'],
  'saw4' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V72L100,120L200,72L300,120L400,72L500,120L600,72L700,120L800,72L900,120L1000,72L1100,120L1200,72V0Z'],
  'saw5' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V70L85.71,120L171.43,70L257.14,120L342.86,70L428.57,120L514.29,70L600,120L685.71,70L771.43,120L857.14,70L942.86,120L1028.57,70L1114.29,120L1200,70V0Z'],
  'saw6' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V68L75,120L150,68L225,120L300,68L375,120L450,68L525,120L600,68L675,120L750,68L825,120L900,68L975,120L1050,68L1125,120L1200,68V0Z'],
  'saw7' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V66L66.67,120L133.33,66L200,120L266.67,66L333.33,120L400,66L466.67,120L533.33,66L600,120L666.67,66L733.33,120L800,66L866.67,120L933.33,66L1000,120L1066.67,66L1133.33,120L1200,66V0Z'],
  'saw8' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V64L60,120L120,64L180,120L240,64L300,120L360,64L420,120L480,64L540,120L600,64L660,120L720,64L780,120L840,64L900,120L960,64L1020,120L1080,64L1140,120L1200,64V0Z'],
  'saw9' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V80L100,110L200,80L300,110L400,80L500,110L600,80L700,110L800,80L900,110L1000,80L1100,110L1200,80V0Z'],
  'saw10' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V78L75,112L150,78L225,112L300,78L375,112L450,78L525,112L600,78L675,112L750,78L825,112L900,78L975,112L1050,78L1125,112L1200,78V0Z'],
  'saw11' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V85L120,115L240,85L360,115L480,85L600,115L720,85L840,115L960,85L1080,115L1200,85V0Z'],
  'saw12' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V90L150,118L300,90L450,118L600,90L750,118L900,90L1050,118L1200,90V0Z'],
  // Scallops 1-8
  'scallop1' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V80Q100,120,200,80Q300,120,400,80Q500,120,600,80Q700,120,800,80Q900,120,1000,80Q1100,120,1200,80V0Z'],
  'scallop2' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V78Q75,120,150,78Q225,120,300,78Q375,120,450,78Q525,120,600,78Q675,120,750,78Q825,120,900,78Q975,120,1050,78Q1125,120,1200,78V0Z'],
  'scallop3' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V76Q60,120,120,76Q180,120,240,76Q300,120,360,76Q420,120,480,76Q540,120,600,76Q660,120,720,76Q780,120,840,76Q900,120,960,76Q1020,120,1080,76Q1140,120,1200,76V0Z'],
  'scallop4' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V74Q50,120,100,74Q150,120,200,74Q250,120,300,74Q350,120,400,74Q450,120,500,74Q550,120,600,74Q650,120,700,74Q750,120,800,74Q850,120,900,74Q950,120,1000,74Q1050,120,1100,74Q1150,120,1200,74V0Z'],
  'scallop5' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V85Q75,115,150,85Q225,115,300,85Q375,115,450,85Q525,115,600,85Q675,115,750,85Q825,115,900,85Q975,115,1050,85Q1125,115,1200,85V0Z'],
  'scallop6' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V84Q60,114,120,84Q180,114,240,84Q300,114,360,84Q420,114,480,84Q540,114,600,84Q660,114,720,84Q780,114,840,84Q900,114,960,84Q1020,114,1080,84Q1140,114,1200,84V0Z'],
  'scallop7' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V82Q50,112,100,82Q150,112,200,82Q250,112,300,82Q350,112,400,82Q450,112,500,82Q550,112,600,82Q650,112,700,82Q750,112,800,82Q850,112,900,82Q950,112,1000,82Q1050,112,1100,82Q1150,112,1200,82V0Z'],
  'scallop8' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V78Q37.50,110,75,78Q112.50,110,150,78Q187.50,110,225,78Q262.50,110,300,78Q337.50,110,375,78Q412.50,110,450,78Q487.50,110,525,78Q562.50,110,600,78Q637.50,110,675,78Q712.50,110,750,78Q787.50,110,825,78Q862.50,110,900,78Q937.50,110,975,78Q1012.50,110,1050,78Q1087.50,110,1125,78Q1162.50,110,1200,78V0Z'],
  // Notches 1-8
  'notch1' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V80H500L600,120L700,80H1200V0Z'],
  'notch2' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V70H420L600,120L780,70H1200V0Z'],
  'notch3' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V80H250L350,120L450,80H750L850,120L950,80H1200V0Z'],
  'notch4' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V80H470C520,80,540,120,600,120C660,120,680,80,730,80H1200V0Z'],
  'notch5' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V75H460Q600,125,740,75H1200V0Z'],
  'notch6' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V70H200L260,120L320,70H1200V0Z'],
  'notch7' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V70H880L940,120L1000,70H1200V0Z'],
  'notch8' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V85H160L200,120L240,85H560L600,120L640,85H960L1000,120L1040,85H1200V0Z'],
  // Arcs 3-8
  'arc3' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V90Q600,150,1200,90V0Z'],
  'arc4' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V110Q600,30,1200,110V0Z'],
  'arc5' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V80Q600,180,1200,80V0Z'],
  'arc6' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V100Q600,0,1200,100V0Z'],
  'arc7' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V90C250,160,520,40,760,130C940,165,1080,115,1200,90V0Z'],
  'arc8' => ['viewBox' => '0 0 1200 120', 'path' => 'M0,0V110C240,35,520,165,780,85C960,30,1100,55,1200,110V0Z'],

];

$sanitizeGradient = function (string $code, string $id): string {
  $code = trim($code);
  if ($code === '') {
    return '';
  }

  // Basic safety: strip defs wrapper, scripts and event handlers.
  $code = preg_replace('~<\s*/?\s*defs\b[^>]*>~i', '', $code);
  $code = preg_replace('~<\s*script\b[^>]*>.*?<\s*/\s*script\s*>~is', '', $code);
  $code = preg_replace('/\son\w+\s*=\s*(\"[^\"]*\"|\'[^\']*\'|[^\s>]+)/i', '', $code);

  // If a gradient tag exists, take the first one. Otherwise treat as <stop>...</stop> list.
  if (preg_match('~<(linearGradient|radialGradient)\b[^>]*>.*?</\1>~is', $code, $m)) {
    $g = $m[0];
  } else {
    $g = '<linearGradient gradientUnits="userSpaceOnUse" x1="0" y1="0" x2="1200" y2="0">' . $code . '</linearGradient>';
  }

  // Force a unique id (we overwrite any id the user provided).
  $g = preg_replace('/\sid\s*=\s*(\"[^\"]*\"|\'[^\']*\')/i', '', $g, 1);
  $g = preg_replace('~^<\s*(linearGradient|radialGradient)\b~i', '<$1 id="' . $id . '"', $g, 1);

  return $g;
};

$sanitizeShape = function (string $markup): string {
    $m = trim($markup);
    if ($m === '') {
        return '';
    }

    // Remove any outer <svg> wrappers (we provide our own <svg>)
    $m = preg_replace('~</?svg\b[^>]*>~i', '', $m);

    // Strip dangerous/unsupported blocks
    $m = preg_replace('~<script\b[^>]*>.*?</script>~is', '', $m);
    $m = preg_replace('~<foreignObject\b[^>]*>.*?</foreignObject>~is', '', $m);
    $m = preg_replace('~<style\b[^>]*>.*?</style>~is', '', $m);
    $m = preg_replace('~<defs\b[^>]*>.*?</defs>~is', '', $m);

    // Remove event handler attributes (onload, onclick, ...)
    $m = preg_replace('~\son[a-z]+\s*=\s*(?:"[^"]*"|\'[^\']*\'|[^\s>]+)~i', '', $m);

    // Remove href / xlink:href
    $m = preg_replace('~\s(?:xlink:href|href)\s*=\s*(?:"[^"]*"|\'[^\']*\')~i', '', $m);

    // Remove javascript: URLs in any attribute value
    $m = preg_replace('~(=)\s*(?:"javascript:[^"]*"|\'javascript:[^\']*\')~i', '$1""', $m);

    return trim($m);
};

$renderSvg = function (
  string $style,
  string $solid,
  string $gradientCode,
  string $gradId,
  string $customShape = ''
) use ($shapes, $sanitizeGradient, $sanitizeShape) {

  if (!isset($shapes[$style])) {
    $style = 'wave1';
  }

  $viewBox = $shapes[$style]['viewBox'];
  $path    = $shapes[$style]['path'];


  // Custom shape (user-provided SVG markup inside <svg>)
  $customShape = trim($customShape);
  if ($customShape !== '') {
    $inner = $sanitizeShape($customShape);

    if ($inner !== '') {

      // Determine fill
      $gradientCode = trim($gradientCode);
      $defs = '';
      $fill = $solid;

      if ($gradientCode !== '') {
        $defs = $sanitizeGradient($gradientCode, $gradId);
        if ($defs !== '') {
          $fill = 'url(#' . $gradId . ')';
        }
      }

      $fillEsc = htmlspecialchars($fill, ENT_QUOTES, 'UTF-8');

      $styleCss =
        '.sc-shape path, .sc-shape polygon, .sc-shape rect, .sc-shape circle, .sc-shape ellipse {' .
        ' fill: ' . $fillEsc . ' !important;' .
        '}';

      return '<svg class="sc-svg" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 1200 120" preserveAspectRatio="none" aria-hidden="true" focusable="false">'
           . ($defs !== '' ? '<defs>' . $defs . '</defs>' : '')
           . '<style><![CDATA[' . $styleCss . ']]></style>'
           . '<g class="sc-shape">' . $inner . '</g>'
           . '</svg>';
    }
  }

  // Gradient (user-provided SVG code)
  $gradientCode = trim($gradientCode);
  if ($gradientCode !== '') {
    $defs = $sanitizeGradient($gradientCode, $gradId);

    if ($defs !== '') {
      $idEsc   = htmlspecialchars($gradId, ENT_QUOTES, 'UTF-8');
      $fill    = 'url(#' . $gradId . ')';
      $fillEsc = htmlspecialchars($fill, ENT_QUOTES, 'UTF-8');

      return '<svg class="sc-svg" xmlns="http://www.w3.org/2000/svg" viewBox="' . $viewBox . '" preserveAspectRatio="none" aria-hidden="true" focusable="false">'
           . '<defs>' . $defs . '</defs>'
           . '<path d="' . $path . '" fill="' . $fillEsc . '" style="fill: ' . $fillEsc . ' !important;" />'
           . '</svg>';
    }
  }

  // Solid (fallback)
  $fillEsc = htmlspecialchars($solid, ENT_QUOTES, 'UTF-8');

  return '<svg class="sc-svg" xmlns="http://www.w3.org/2000/svg" viewBox="' . $viewBox . '" preserveAspectRatio="none" aria-hidden="true" focusable="false">'
       . '<path d="' . $path . '" fill="' . $fillEsc . '" style="fill: ' . $fillEsc . ' !important;" />'
       . '</svg>';
};
$topSize     = max(0, (int) ($props['top_size'] ?? 0));
$topSizeS    = max(0, (int) ($props['top_size_s'] ?? 0)) ?: $topSize;
$bottomSize  = max(0, (int) ($props['bottom_size'] ?? 0));
$bottomSizeS = max(0, (int) ($props['bottom_size_s'] ?? 0)) ?: $bottomSize;

// If no explicit height is set, ensure the wrapper has a min-height so absolute dividers are visible.
$minH = 0;
if ($heightVal <= 0) {
    if ($top) {
        $minH = max($minH, (int) ($props['top_offset'] ?? 0) + $topSize);
    }
    if ($bottom) {
        $minH = max($minH, (int) ($props['bottom_offset'] ?? 0) + $bottomSize);
    }}
?>

<?= $wrap($props, $attrs) ?>

<style>
.<?= $this->escape($uid) ?>{
  /* Variables */
  --sc-height: <?= $heightVal > 0 ? $heightVal . $heightUnit : 'auto' ?>;
  --sc-z: <?= (int) $zIndex ?>;

  --sc-top-size: <?= $topSize ?>px;
  --sc-top-size-s: <?= $topSizeS ?>px;
  --sc-top-offset: <?= (int) ($props['top_offset'] ?? 0) ?>px;
  --sc-top-rot: <?= (float) ($props['top_rotate'] ?? 0) ?>deg;
  --sc-top-color: <?= $this->escape($topColor) ?>;

  --sc-bottom-size: <?= $bottomSize ?>px;
  --sc-bottom-size-s: <?= $bottomSizeS ?>px;
  --sc-bottom-offset: <?= (int) ($props['bottom_offset'] ?? 0) ?>px;
  --sc-bottom-rot: <?= (float) ($props['bottom_rotate'] ?? 0) ?>deg;
  --sc-bottom-color: <?= $this->escape($bottomColor) ?>;

  --sc-top-shadow: <?= $topShadowCss ?>;
  --sc-bottom-shadow: <?= $bottomShadowCss ?>;
  position: relative;
  width: 100%;
  height: var(--sc-height);
  z-index: var(--sc-z);
  <?= $overflowHidden ? 'overflow:hidden;' : '' ?>
  <?= $minH > 0 ? 'min-height:' . (int) $minH . 'px;' : '' ?>
  <?= $bg !== '' ? 'background:' . str_replace(['</style', '</STYLE'], '', $bg) . ';' : '' ?>
}

/* Base */
.<?= $this->escape($uid) ?> .sc-divider{
  position: absolute;
  pointer-events: none;
  line-height: 0;
  overflow: visible;
}

.<?= $this->escape($uid) ?> .sc-svg{
  display:block;
  width:100%;
  height:100%;
}


/* Force fill via variables (more reliable than currentColor in some setups) */
.<?= $this->escape($uid) ?> .sc-top:not(.sc-has-gradient) .sc-svg path{ fill: var(--sc-top-color, currentColor) !important; }
.<?= $this->escape($uid) ?> .sc-bottom:not(.sc-has-gradient) .sc-svg path{ fill: var(--sc-bottom-color, currentColor) !important; }
/* Top */
.<?= $this->escape($uid) ?> .sc-top{
  left:0; right:0;
  top: var(--sc-top-offset);
  height: var(--sc-top-size);
  color: var(--sc-top-color, currentColor) !important;
}

.<?= $this->escape($uid) ?> .sc-top .sc-svg{
  transform: rotate(var(--sc-top-rot));
  transform-origin: center;
  filter: var(--sc-top-shadow, none) !important;
}

/* Bottom (auto flipped) */
.<?= $this->escape($uid) ?> .sc-bottom{
  left:0; right:0;
  bottom: var(--sc-bottom-offset);
  height: var(--sc-bottom-size);
  color: var(--sc-bottom-color, currentColor) !important;
}
.<?= $this->escape($uid) ?> .sc-bottom .sc-svg{
  transform: rotate(var(--sc-bottom-rot)) scaleY(-1);
  transform-origin: center;
  filter: var(--sc-bottom-shadow, none) !important;
}
/* Mobile sizes */
@media (max-width: 640px){
  .<?= $this->escape($uid) ?> .sc-top{ height: var(--sc-top-size-s); 
}
  .<?= $this->escape($uid) ?> .sc-bottom{ height: var(--sc-bottom-size-s); }
}
</style>

<?php if ($top) : ?>
  <div class="sc-divider sc-top<?= $topHasGradient ? ' sc-has-gradient' : '' ?>"><?= $renderSvg((string) ($props['top_style'] ?? 'wave1'), $topColor, $topGradientCode, $uid . '_top_grad', ((string) ($props['top_style'] ?? 'wave1') === 'custom') ? $topCustomSvg : '') ?></div>
<?php endif; ?>

<?php if ($bottom) : ?>
  <div class="sc-divider sc-bottom<?= $bottomHasGradient ? ' sc-has-gradient' : '' ?>"><?= $renderSvg((string) ($props['bottom_style'] ?? 'wave1'), $bottomColor, $bottomGradientCode, $uid . '_bottom_grad', ((string) ($props['bottom_style'] ?? 'wave1') === 'custom') ? $bottomCustomSvg : '') ?></div>
<?php endif; ?>

<?php if ($left) : ?>
<?php endif; ?>

<?php if ($right) : ?>
<?php endif; ?>

<?= $wrap->end() ?>
